<?php

namespace YMFSEO;

// Exits if accessed directly.
if ( ! \defined( 'ABSPATH' ) ) exit;

/**
 * Provides plugin settings functionality.
 * 
 * @since 2.0.0
 */
class Settings {
	/**
	 * Settings module parameters.
	 * 
	 * @var array
	 */
	public static array $params = [];
	
	/**
	 * Default YM Fast SEO settings options values.
	 * 
	 * @var array
	 */
	public static array $default_settings = [
		'hide_title_parts'           => true,
		'title_separator'            => '|',
		'clear_excerpts'             => true,
		'hide_users_sitemap'         => true,
		'post_type_page_type_page'   => 'WebPage',
		'preview_image_id'           => 0,
		'preview_size'               => 'summary_large_image',
		'rep_type'                   => 'org',
		'rep_org_type'               => 'Organization',
		'rep_org_name'               => '',
		'rep_person_name'            => '',
		'rep_email'                  => '',
		'rep_phone'                  => '',
		'rep_org_city'               => '',
		'rep_org_region'             => '',
		'rep_org_address'            => '',
		'rep_org_postal_code'        => '',
		'rep_image_id'               => 0,
		'google_search_console_key'  => '',
		'bing_webmaster_tools_key'   => '',
		'yandex_webmaster_key'       => '',
		'indexnow_key'               => '',
		'indexnow_enabled'           => true,
		'redirects'                  => [],
		'head_scripts'               => '',
		'head_scripts_only_visitors' => true,
		'robots_txt'                 => '',
		'enable_llms_txt'            => false,
	];
	
	/**
	 * Contains info about registered sections.
	 * 
	 * @var array
	 */
	public static array $registered_sections = [];

	/**
	 * Inits YMFSEO Settings.
	 * 
	 * @since 3.0.0
	 */
	public static function init () {
		// Registers YM Fast SEO settings page.
		add_action( 'admin_menu', function () {
			add_options_page(
				Settings::$params[ 'page_title' ],
				Settings::$params[ 'menu_label' ],
				Settings::$params[ 'capability' ],
				Settings::$params[ 'page_slug' ],
				fn () => include YMFSEO_ROOT_DIR . 'parts/settings-page.php',
				Settings::$params[ 'menu_position' ]
			);
		});

		// Adds YM Fast SEO settings sections and options.
		add_action( 'admin_init', function () {
			/**
			 * General section.
			 */

			/* translators: Settings section name */
			Settings::add_section( 'general', __( 'General', 'ym-fast-seo' ), 'dashicons-admin-settings', [
				/* translators: %s: Link to general settings page */
				'description' => sprintf( wp_kses_post( __( 'To update the site name and description, navigate to the <a href="%s">general settings page</a>.', 'ym-fast-seo' ) ),
					esc_url( get_admin_url( null, 'options-general.php' ) ),
				),
			]);
			Settings::register_option(
				'hide_title_parts',
				/* translators: Verb */
				__( 'Clear Titles', 'ym-fast-seo' ),
				'boolean',
				'general',
				'checkbox',
				[
					'label'       => __( 'Simplify title tags by removing unnecessary parts', 'ym-fast-seo' ),
					'description' => __( 'The site description on the front page, and the site name on all other pages.', 'ym-fast-seo' ),
				],
			);
			Settings::register_option(
				'title_separator',
				__( 'Title Separator', 'ym-fast-seo' ),
				'string',
				'general',
				'separator',
				[
					'options'     => [ '|', '-', '–', '—', ':', '/', '·', '•', '⋆', '~', '«', '»', '<', '>' ],
					
					'description' => sprintf(
						/* translators: %s: Separator tag name */
						__( 'Specify the separator used in titles and %s tag.', 'ym-fast-seo' ),
						'<code>%sep%</code>',
					),
				],
			);
			Settings::register_option(
				'clear_excerpts',
				/* translators: Verb */
				__( 'Clear Excerpts', 'ym-fast-seo' ),
				'boolean',
				'general',
				'checkbox',
				[
					'label'       => __( 'Enhance excerpts by removing unnecessary parts', 'ym-fast-seo' ),
					'description' => __( 'Removes headings from excerpts.', 'ym-fast-seo' ),
				],
			);
			Settings::register_option(
				'hide_users_sitemap',
				/* translators: Verb */
				__( 'Hide Users Sitemap', 'ym-fast-seo' ),
				'boolean',
				'general',
				'checkbox',
				[
					'label' => __( 'Exclude the users page from the sitemap', 'ym-fast-seo' ),
				],
			);

			/**
			 * Post Types section.
			 */

			/* translators: Settings section name */
			Settings::add_section( 'post-types', __( 'Post Types', 'ym-fast-seo' ), 'dashicons-admin-post', [
				'description' => implode( "</p><p>",[
					__( 'The default title and page type values for single post pages.', 'ym-fast-seo' ),
					sprintf(
						/* translators: %s: List of available tags */
						__( 'Available tags: %s.', 'ym-fast-seo' ),
						implode( ', ',[
							'<code>%post_title%</code>',
							...array_map( function ( $tag ) {
								return "<code>{$tag}</code>";
							}, array_keys( MetaFields::$replace_tags ) ),
						]),
					),
				]),
			]);
			foreach ( Core::get_public_post_types( 'objects' ) as $post_type ) {
				$post_type_tags = array_map( function ( string $tag ) : string {
					return "<code>{$tag}</code>";
				}, array_keys( apply_filters( "ymfseo_{$post_type->name}_posts_tags", [], 0 ) ) );

				Settings::register_option(
					"post_type_title_{$post_type->name}",
					$post_type->label,
					'string',
					'post-types',
					'text',
					[
						'placeholder' => '%post_title%',
						'menu_icon'   => $post_type->menu_icon,
						'description' => $post_type_tags ? sprintf(
							/* translators: %s: List of available tags */
							__( 'Custom tags: %s.', 'ym-fast-seo' ),
							implode( ', ', $post_type_tags ),
						) : '',
					],
				);
				Settings::register_option(
					"post_type_page_type_{$post_type->name}",
					__( 'Page Type', 'ym-fast-seo' ),
					'string',
					'post-types',
					'select',
					[
						'class'   => 'sub-field',
						'options' => Core::$page_types,
					],
				);
			}

			/**
			 * Archives section.
			 */

			$post_types_with_archives = Core::get_post_types_with_archives();

			/* translators: Settings section name */
			Settings::add_section( 'archives', __( 'Archives', 'ym-fast-seo' ), 'dashicons-archive', [
				'description' => count( $post_types_with_archives )
					? ''
					: __( 'Your site doesn’t currently have any post types that support archives.', 'ym-fast-seo' ),
			]);

			foreach ( $post_types_with_archives as $post_type ) {
				Settings::register_option(
					"archive_title_{$post_type->name}",
					$post_type->label,
					'string',
					'archives',
					'text',
					[
						'menu_icon' => $post_type->menu_icon,
					],
				);
				Settings::register_option(
					"archive_description_{$post_type->name}",
					__( 'Description', 'ym-fast-seo' ),
					'string',
					'archives',
					'textarea',
					[
						'class' => 'sub-field',
					],
				);
			}

			/**
			 * Taxonomies section.
			 */

			/* translators: Settings section name */
			Settings::add_section( 'taxonomies', __( 'Taxonomies', 'ym-fast-seo' ), 'dashicons-tag', [
				'description' => implode( "</p><p>",[
					__( 'The default title and description values for taxonomy term pages.', 'ym-fast-seo' ),
					sprintf(
						/* translators: %s: List of available tags */
						__( 'Available tags: %s.', 'ym-fast-seo' ),
						implode( ', ',[
							'<code>%term_title%</code>',
							...array_map( function ( $tag ) {
								return "<code>{$tag}</code>";
							}, array_keys( MetaFields::$replace_tags ) ),
						]),
					),
				]),
			]);
			foreach ( Core::get_public_taxonomies( 'objects' ) as $taxonomy ) {
				$taxonomy_tags = array_map( function ( string $tag ) : string {
					return "<code>{$tag}</code>";
				}, array_keys( apply_filters( "ymfseo_{$taxonomy->name}_taxonomy_tags", [], 0 ) ) );

				Settings::register_option(
					"taxonomy_title_{$taxonomy->name}",
					$taxonomy->label,
					'string',
					'taxonomies',
					'text',
					[
						'placeholder' => '%term_title%',
					],
				);
				Settings::register_option(
					"taxonomy_description_{$taxonomy->name}",
					__( 'Description', 'ym-fast-seo' ),
					'string',
					'taxonomies',
					'textarea',
					[
						'class'       => 'sub-field',
						'description' => $taxonomy_tags ? sprintf(
							/* translators: %s: List of available tags */
							__( 'Custom tags: %s.', 'ym-fast-seo' ),
							implode( ', ', $taxonomy_tags ),
						) : '',
					],
				);
				Settings::register_option(
					"taxonomy_noindex_{$taxonomy->name}",
					__( 'Indexing', 'ym-fast-seo' ),
					'boolean',
					'taxonomies',
					'checkbox',
					[
						'class' => 'sub-field',
						'label' => __( 'Disallow indexing', 'ym-fast-seo' ),
					],
				);
			}

			/**
			 * Site Preview section.
			 */

			/* translators: Settings section name */
			Settings::add_section( 'preview', __( 'Site Preview', 'ym-fast-seo' ), 'dashicons-format-image' );
			Settings::register_option(
				'preview_image_id',
				/* translators: Noun */
				__( 'Preview Image', 'ym-fast-seo' ),
				'integer',
				'preview',
				'image',
				[
					'description' => sprintf(
						/* translators: %s: Size in pixels */
						__( 'The image link will be added to the meta tags if no post/page thumbnail is set. The recommended size is %s pixels.', 'ym-fast-seo' ),
						'<code>1200 × 630</code>',
					),
				],
			);
			Settings::register_option(
				'preview_size',
				/* translators: Noun */
				__( 'Preview Size', 'ym-fast-seo' ),
				'string',
				'preview',
				'select',
				[
					'options' => [
						/* translators: Twitter Cards property */
						'summary'             => __( 'Summary', 'ym-fast-seo' ),
						/* translators: Twitter Cards property */
						'summary_large_image' => __( 'Large Image', 'ym-fast-seo' ),
					],
				],
			);

			/**
			 * Representative section.
			 */

			/* translators: Settings section name */
			Settings::add_section( 'representative', __( 'Representative', 'ym-fast-seo' ), 'dashicons-businessperson', [
				'description' => __( 'If this website represents a company or person, you can include some details. This information will not be visible to visitors but will be available to search engines.', 'ym-fast-seo' ),
			]);
			Settings::register_option(
				'rep_type',
				__( 'Represented by', 'ym-fast-seo' ),
				'string',
				'representative',
				'select',
				[
					'class'   => 'rep-type',
					'options' => [
						'org'    => __( 'Organization', 'ym-fast-seo' ),
						'person' => __( 'Person', 'ym-fast-seo' ),
					],
				],
			);
			Settings::register_option(
				'rep_org_type',
				__( 'Organization Type', 'ym-fast-seo' ),
				'string',
				'representative',
				'select',
				[
					'class'   => 'rep-org',
					'options' => [
						/* translators: Organization type */
						'Organization'            => __( 'Regular', 'ym-fast-seo' ),
						/* translators: Organization type */
						'LocalBusiness'           => __( 'Local Business', 'ym-fast-seo' ),
						/* translators: Organization type */
						'OnlineBusiness'          => __( 'Online Business', 'ym-fast-seo' ),
						/* translators: Organization type */
						'OnlineStore'             => __( 'Online Store', 'ym-fast-seo' ),
						/* translators: Organization type */
						'NewsMediaOrganization'   => __( 'News/Media', 'ym-fast-seo' ),
						/* translators: Organization type */
						'MedicalOrganization'     => __( 'Medical', 'ym-fast-seo' ),
						/* translators: Organization type */
						'EducationalOrganization' => __( 'Educational', 'ym-fast-seo' ),
						/* translators: Organization type */
						'SportsOrganization'      => __( 'Sports', 'ym-fast-seo' ),
						/* translators: Organization type */
						'MusicGroup'              => __( 'Music Group', 'ym-fast-seo' ),
						/* translators: Organization type */
						'NGO'                     => __( 'Non-Governmental', 'ym-fast-seo' ),
						/* translators: Organization type */
						'Project'                 => __( 'Project', 'ym-fast-seo' ),
					],
				],
			);
			Settings::register_option(
				'rep_org_name',
				__( 'Organization Name', 'ym-fast-seo' ),
				'string',
				'representative',
				'text',
				[
					'class'        => 'rep-org',
					'autocomplete' => 'organization',
				],
			);
			Settings::register_option(
				'rep_person_name',
				__( 'Person Name', 'ym-fast-seo' ),
				'string',
				'representative',
				'text',
				[
					'class'        => 'rep-person',
					'autocomplete' => 'name',
				],
			);
			Settings::register_option(
				'rep_email',
				__( 'Email', 'ym-fast-seo' ),
				'string',
				'representative',
				'text',
				[
					'type'         => 'email',
					'autocomplete' => 'email',
				],
			);
			Settings::register_option(
				'rep_phone',
				__( 'Phone', 'ym-fast-seo' ),
				'string',
				'representative',
				'text',
				[
					'type'         => 'tel',
					'autocomplete' => 'tel',
				],
			);
			Settings::register_option(
				'rep_org_city',
				__( 'City', 'ym-fast-seo' ),
				'string',
				'representative',
				'text',
				[
					'class' => 'rep-org',
				],
			);
			Settings::register_option(
				'rep_org_region',
				__( 'Region', 'ym-fast-seo' ),
				'string',
				'representative',
				'text',
				[
					'class' => 'rep-org',
				],
			);
			Settings::register_option(
				'rep_org_address',
				__( 'Address', 'ym-fast-seo' ),
				'string',
				'representative',
				'text',
				[
					'class' => 'rep-org',
				],
			);
			Settings::register_option(
				'rep_org_postal_code',
				__( 'Postal Code', 'ym-fast-seo' ),
				'string',
				'representative',
				'text',
				[
					'class'       => 'rep-org',
					'input-class' => 'code',
				],
			);
			Settings::register_option(
				'rep_image_id',
				__( 'Image', 'ym-fast-seo' ),
				'integer',
				'representative',
				'image',
				[
					'description' => __( 'The representative\'s image will be available to search engines.', 'ym-fast-seo' ),
				],
			);

			/**
			 * Integrations section.
			 */

			/* translators: Settings section name */
			Settings::add_section( 'integrations', __( 'Integrations', 'ym-fast-seo' ), 'dashicons-rest-api', [
				'description' => sprintf(
					/* translators: %s: <meta> tag `content` attribute name */
					__( 'Enter the verification codes for the required services. They are usually found in the %s attribute of the verification meta tag.', 'ym-fast-seo' ),
					'<code>content</code>',
				),
			]);
			Settings::register_option(
				'google_search_console_key',
				sprintf( '<a href="https://search.google.com/search-console" target="_blank">%s</a>',
					/* translators: Service name (probably doesn't translate) */
					__( 'Google Search Console', 'ym-fast-seo' ),
				),
				'string',
				'integrations',
				'text',
				[
					'input-class' => 'code',
				],
			);
			Settings::register_option(
				'bing_webmaster_tools_key',
				sprintf( '<a href="https://www.bing.com/webmasters" target="_blank">%s</a>',
					/* translators: Service name (probably doesn't translate) */
					__( 'Bing Webmaster Tools', 'ym-fast-seo' ),
				),
				'string',
				'integrations',
				'text',
				[
					'input-class' => 'code',
				],
			);
			Settings::register_option(
				'yandex_webmaster_key',
				sprintf( '<a href="https://webmaster.yandex.ru/" target="_blank">%s</a>',
					/* translators: Service name (probably doesn't translate) */
					__( 'Yandex Webmaster', 'ym-fast-seo' ),
				),
				'string',
				'integrations',
				'text',
				[
					'input-class' => 'code',
				],
			);
			Settings::register_option(
				'indexnow_key',
				/* translators: IndexNow – Protocol name (probably doesn't translate) */
				__( 'IndexNow Key', 'ym-fast-seo' ),
				'string',
				'integrations',
				'text',
				[
					'input-class' => 'code',
					'readonly'    => true,
					'description' => __( 'IndexNow API key is generated automatically.', 'ym-fast-seo' ),
				],
			);
			Settings::register_option(
				'indexnow_enabled',
				/* translators: Option name */
				__( 'IndexNow Sending', 'ym-fast-seo' ),
				'boolean',
				'integrations',
				'checkbox',
				[
					'label'       => __( 'Enable IndexNow sending', 'ym-fast-seo' ),
					'description' => implode([
						sprintf( Checker::is_site_public()
							? ''
							/* translators: %s: Link to settings page */
							: '<span class="dashicons dashicons-warning"></span> ' . __( 'The site is configured to <a href="%s">discourage indexing</a>, IndexNow is disabled regardless of this option.', 'ym-fast-seo' ),
							get_admin_url( null, 'options-reading.php#blog_public' ),
						),
					]),
				],
			);

			/**
			 * Redirects section.
			 */

			/* translators: Redirects section name */
			// Settings::add_section( 'redirects', __( 'Redirects', 'ym-fast-seo' ), 'dashicons-randomize' );
			// Settings::register_option(
			// 	'redirects',
			// 	__( 'Redirects', 'ym-fast-seo' ),
			// 	'array',
			// 	'redirects',
			// 	'redirects',
			// );

			/**
			 * Additional section.
			 */

			/* translators: Settings section name */
			Settings::add_section( 'additional', __( 'Additional', 'ym-fast-seo' ), 'dashicons-admin-generic' );
			Settings::register_option(
				'head_scripts',
				__( 'Head Scripts', 'ym-fast-seo' ),
				'string',
				'additional',
				'textarea',
				[
					'rows'        => 8,
					'codemirror'  => true,
					'description' => sprintf(
						/* translators: %s: <head> tag name */
						__( 'Here you can insert analytics counters code and other scripts. The code will be printed inside the %s tag.', 'ym-fast-seo' ),
						'<code>&lt;head&gt;</code>',
					),
				],
			);
			Settings::register_option(
				'head_scripts_only_visitors',
				__( 'Only for Visitors', 'ym-fast-seo' ),
				'boolean',
				'additional',
				'checkbox',
				[
					'label' => __( 'Do not include head scripts for logged-in users', 'ym-fast-seo' ),
				],
			);
			Settings::register_option(
				'robots_txt',
				sprintf(
					/* translators: %s: robots.txt */
					__( 'Edit %s', 'ym-fast-seo' ),
					'robots.txt',
				),
				'string',
				'additional',
				'robots-txt',
			);
			Settings::register_option(
				'enable_llms_txt',
				/* translators: %s: llms.txt */
				sprintf( __( 'Enable %s', 'ym-fast-seo' ), 
					esc_html( 'llms.txt' ),
				),
				'boolean',
				'additional',
				'checkbox',
				[
					'label' => __( 'Enable generation', 'ym-fast-seo' ),
					/* translators: %1$s: llms.txt, %2$s: llms-full.txt */
					'description' => sprintf( __( 'Creates automatically generated %1$s and %2$s files used by AI crawlers.', 'ym-fast-seo' ),
						wp_kses_post( '<code>llms.txt</code>' ),
						wp_kses_post( '<code>llms-full.txt</code>' ),
					),
				],
			);
		});
	}

	/**
	 * Adds section to YM Fast SEO settings page.
	 * 
	 * @since 3.2.0 Has `$icon` argument.
	 * 
	 * @param string $slug  Section slug.
	 * @param string $title Section title.
	 * @param string $icon  Full dashicon name e.g. `dashicons-admin-generic`.
	 * @param array  $args {
	 * 		Section arguments.
	 * 
	 * 		@type string $description Section description below title. 
	 * }
	 */
	public static function add_section ( string $slug, string $title, string $icon, array $args = [] ) {
		Settings::$registered_sections[] = [
			'slug'  => $slug,
			'title' => $title,
			'icon'  => $icon,
		];

		add_settings_section( "ymfseo_{$slug}_section",
			"<span class=\"dashicons {$icon}\"></span> {$title}",
			fn ( $args ) => include YMFSEO_ROOT_DIR . 'parts/settings-section.php',
			Settings::$params[ 'page_slug' ],
			[
				'after_section' => implode( ' ', [
					sprintf( '<div class="ymfseo-submit"><button class="%s">%s</button></div>',
						esc_attr( 'button button-primary' ),
						esc_attr__( 'Save Changes', 'ym-fast-seo' ),
					),
					sprintf( '</section><section>' ),
				]),
				...$args,
			]
		);
	}

	/**
	 * Prepares option name before usage.
	 * 
	 * @since 3.0.0
	 * 
	 * @param string $option Option name.
	 */
	public static function prepare_option_name ( string &$option ) {
		if ( 'ymfseo_' !== mb_substr( $option, 0, 7 ) ) {
			$option = "ymfseo_{$option}";
		}
	}

	/**
	 * Registers YM Fast SEO settings option.
	 * 
	 * @param string $slug       Option name.
	 * @param string $title      Option title.
	 * @param string $type       Option type.
	 * @param string $section    Option section slug without 'ymfseo_$_section'.
	 * @param string $field_part Option field part file name without 'parts/settings-$-field.php'.
	 * @param array  $args {
	 * 		Option arguments.
	 * 
	 * 		@type string $menu_icon Menu dashicon name.
	 * }
	 */
	public static function register_option ( string $slug, string $title, string $type, string $section, string $field_part, array $args = [] ) {
		// Checks is default value exist.
		if ( ! isset( Settings::$default_settings[ $slug ] ) ) {
			$break   = true;
			$default = '';

			// Allowed settings by mask. Slug mask => default value.
			$allowed = [
				'post_type_title_'       => '',
				'post_type_page_type_'   => 'ItemPage',
				'archive_title_'         => '',
				'archive_description_'   => '',
				'taxonomy_title_'        => '',
				'taxonomy_description_'  => '',
				'taxonomy_noindex_'      => false,
			];

			// Checks is setting allowed by mask.
			foreach ( $allowed as $allowed_item => $default_value ) {
				if ( str_contains( $slug, $allowed_item ) ) {
					$break   = false;
					$default = $default_value;
					
					break;
				}
			}

			// Breaks if not allowed.
			if ( $break ) return;

			// Begins setting init.
			Settings::$default_settings[ $slug ] = $default;
		}

		// Defines sanitize callback.
		$sanitize_callback = 'sanitize_text_field';

		if ( \in_array( $slug, [ 'head_scripts', 'robots_txt' ] ) ) {
			$sanitize_callback = function ( $value ) {
				return wp_unslash( $value );
			};
		}

		if ( 'redirects' == $slug ) {
			$sanitize_callback = function ( $value ) : array {
				foreach ( $value as $i => $item ) {
					foreach ( $item as $param ) {
						if ( empty( $param ) && isset( $value[ $i ] ) ) {
							unset( $value[ $i ] );
						}
					}
				}

				if ( ! is_array( $value ) ) {
					return [];
				}

				return $value;
			};
		}

		$tagged_options = [
			'post_type_title_', 'post_type_description_',
			'taxonomy_title_', 'taxonomy_description_'
		];
		
		foreach ( $tagged_options as $option_slug ) {
			if ( str_contains( $slug, $option_slug ) ) {
				$sanitize_callback = [ '\YMFSEO\Core', 'sanitize_text_field' ];
			}
		}

		// Check menu icon.
		$menu_icon = false;

		if ( isset( $args[ 'menu_icon' ] ) ) {
			$menu_icon = $args[ 'menu_icon' ];

			if ( ! str_contains( $menu_icon, 'dashicons-' ) ) {
				$menu_icon = false;
			}
		}

		// Registers setting.
		// phpcs:ignore
		register_setting( Settings::$params[ 'page_slug' ], "ymfseo_{$slug}", [
			'type'              => $type,
			'default'           => Settings::$default_settings[ $slug ],
			'sanitize_callback' => $sanitize_callback,
		]);

		// Adds field.
		add_settings_field( "ymfseo_{$slug}",
			sprintf( '%s %s',
				$menu_icon ? "<span class=\"dashicons {$menu_icon}\"></span>" : '',
				$title,
			),
			fn ( $args ) => include YMFSEO_ROOT_DIR . "parts/settings-{$field_part}-field.php",
			Settings::$params[ 'page_slug' ],
			"ymfseo_{$section}_section",
			array_merge( [ 'label_for' => "ymfseo_{$slug}" ], $args )
		);
	}

	/**
	 * Retrieves an YMFSEO option value based on an option name.
	 * 
	 * @since 2.1.0 Has `$default` argument.
	 * 
	 * @param string $option  Option name. Allowed without 'ymfseo_'.
	 * @param mixed  $default Default option value.
	 * 
	 * @return mixed Option or default value.
	 */
	public static function get_option ( string $option, mixed $default = false ) : mixed {
		Settings::prepare_option_name( $option );

		$default_value = Settings::$default_settings[ str_replace( 'ymfseo_', '', $option ) ] ?? $default;

		return get_option( $option, $default_value );
	}

	/**
	 * Updates option value in database or create new.
	 *
	 * @since 3.0.0
	 *
	 * @param string $option Option name. Allowed without 'ymfseo_'.
	 * @param mixed  $value  Option value.

	 * @return bool `true` if option updated, `false` if no changes or error.
	 */
	public static function update_option ( string $option, mixed $value ) : bool {
		Settings::prepare_option_name( $option );

		return update_option( $option, $value );
	}
}